% PROJECTION_CAPITAL_STDX0P01:
% Solving the capital model with stdx = 0.01 
% 
%  Nicolas Petrosky-Nadeau, Lu Zhang, and Lars Kuehn, January 2018


clear all; close all; clc; format compact; format long 

% Open parallel environment
needNewWorkers = (matlabpool('size') == 0);
if needNewWorkers    
    % Open a new MATLAB pool 
    matlabpool open 16
end

% Define parameter values
xbar    = -.771                     % long-run mean of log productivity
rhox    = .95^(1/3)
stdx    = .01
% 
alpha   = 1/3                       % capital share in production funciton
delta   = .01                       % capital depreciation rate
nu      = 2                         % the elasticity of the investment rate to q
a1      = delta/(1 - nu)
a2      = delta^(1/nu)
%  
beta    = exp( -(log(1.0573))/12 )
eta     = .04                       % worker's bargaining weight
s       = .04
kappa0  = .5       
kappa1  = .5      
iota    = 1.25                      % elasticity of the matching function 
b       = .85                       % unemployment benefits to average labor productivity

% solving for the steady state theta
param_ss  = {kappa0, kappa1, eta, b, s, iota, xbar, rhox, stdx, beta, alpha, delta};
theta0    = 1.8;

opts      = optimset('MaxIter', 100, 'TolFun', 1e-12, 'TolX', 1e-12);
[thetass, fval, exitflag] = fzero('thetass_fn', theta0, opts, param_ss)

if exitflag~=1
    disp('Problem: no thetass with current parameters')
   % disp('Setting thetass manually to 1.25 as in the AP paper...')
    %thetass = 1.25; 
    return
end

qss      = (thetass^iota + 1)^(-1/iota)
fss      = qss*thetass 
uss      = s/(fss + s)
nss      = 1 - uss;
kss      = nss*( ( (alpha*exp(xbar))/(1/beta - 1 + delta) )^(1/(1 - alpha)) )
yss      = exp(xbar)*(kss^alpha)*(nss^(1 - alpha))
MPLss    = (1 - alpha)*yss/nss
MPKss    = alpha*yss/kss
kappatss = kappa0 + kappa1*qss
wss      = eta*((1 - alpha)*yss/nss + kappatss*thetass) + (1 - eta)*b
vss      = thetass*uss;
iss      = delta*kss
css      = yss - kappatss*vss - iss 
Jss      = css; 

% Collect model parameters and steady state values:
params  = {kappa0, kappa1, eta, b, s, iota, xbar, rhox, stdx, beta, alpha, delta, a1, a2, nu};
SS      = {uss, nss, qss, fss, thetass, yss, vss, css, wss, Jss, kss, kappatss};

% Discretize aggregate shocks via the Rouwenhorst (1995) method
nX      = 17                                         
xdev    = 2*stdx/sqrt((1 - rhox^2)*(nX - 1));
[P, x]  = rouwTrans(rhox, xbar, xdev, nX);                
X       = exp(x);

% Ground work for cubic splines
nN      = 10
Nmin    = .65 
Nmax    = .99
% 
nK      = 10
Kmin    = 15
Kmax    = 40

fspace  = fundefn('spli', [nN nK], [Nmin Kmin], [Nmax Kmax]);
[NK, B] = funnode(fspace);
N       = B{1};            % Grid for N: nN by 1 
K       = B{2};            % Grid for K: nK by 1
% 
NKgrid  = NK;
Ngrid   = NKgrid(:, 1);
Kgrid   = NKgrid(:, 2);

% Working matrices: nN*nK by nX 
Xm      = repmat(X', [nN*nK 1]);
Nm      = repmat(Ngrid, [1 nX]);
Km      = repmat(Kgrid, [1 nX]);
Xp      = permute(repmat(Xm, [1 1 nX]), [1 3 2]);
% A more intuitive construction of Xp:
% Xp     = zeros(nNK, nX, nX);
% for ip = 1 : nx
%    Xp(:, :, ip) = X(ip)*ones(nNK, nX);  
% end

% Define a cell array to pass parameters to the residual function
gridsetup = {P, X, N, K, NK, fspace, Nm, Km, Xm, Xp};


load output_proj_capital_stdx0p01  aE aI 
    
a0 = [aE(:); aI(:)];

% Solve for approximating coefficients

% Solving with fsolve
precision = 0
if precision == 0
    opts = optimset('Display', 'Iter', 'MaxIter', 100, 'MaxFunEvals', 500000);
elseif precision == 1
    opts = optimset('Display', 'Iter', 'MaxIter', 500, 'MaxFunEvals', 50000, 'TolFun', 1e-12, 'TolX', 1e-12);
end
[a, fval, exitflag]  = fsolve('residual_capital', a0, opts, params, gridsetup);

exitflag

aE      = reshape(a(1 : nN*nK*nX), [nN*nK nX]);
aI      = reshape(a(nN*nK*nX+1 : 2*nN*nK*nX), [nN*nK nX]);

if exitflag == 1
    save output_proj_capital  aE aI params gridsetup SS
end


% Close parallel environment
if needNewWorkers
    matlabpool close
end




