% PARSIMU_CAPITAL: 
% Simulations for the capital model 
% 
%  Nicolas Petrosky-Nadeau, Lu Zhang, and Lars Kuehn, January 2018


clear all; close all; clc; format compact; format short; 


% Open parallel environment
needNewWorkers = (matlabpool('size') == 0);
if needNewWorkers    
    % Open a new MATLAB pool with 16 workers.
    matlabpool open 16
end


load output_proj_capital
load moments_theory_capital

[kappa0, kappa1, eta, b, s, iota, xbar, rhox, stdx, beta, alpha, delta, a1, a2, nu] = params{:};
[P, X, N, K, NK, fspace, Nm, Km, Xm, Xp] = gridsetup{:};

[uss, nss, qss, fss, thetass, yss, vss, css, wss, Jss, kss, kappatss] = SS{:}; 

nN      = length(N);
nK      = length(K);
nNK     = nN*nK;
nX      = length(X);
%
Nmin    = min(N);
Nmax    = max(N);
Kmin    = min(K);
Kmax    = max(K);

burnin  = 500*12; 
parnsim = 100
nsim    = 100 

% average length of AP data in months
Tap     = 124*12;                                   
Tc      = 138*12;
Ty      = 153*12; 
Tuv     = 1017; 
len     = max([Tap Tc Ty Tuv]); 
% 
% + 1 to make sure (excess) returns have nsim*len periods
T       = burnin + nsim*len + 1


if 1 % change 1 to 0 if you have run the code once and saved the stationary distribution 
U_t    = zeros(len*nsim, parnsim); 
V_t    = zeros(len*nsim, parnsim); 
K_t    = zeros(len*nsim, parnsim); 
I_t    = zeros(len*nsim, parnsim); 
expx_t = zeros(len*nsim, parnsim);  
LPY_t  = zeros(len*nsim, parnsim);  
W_t    = zeros(len*nsim, parnsim);  
C_t    = zeros(len*nsim, parnsim); 
Y_t    = zeros(len*nsim, parnsim); 

tic; 
parfor pn = 1 : parnsim
    
% simulating artificial data in one shot
x            = log(X); 
[x_t, index] = DiscSimu(median(x), x, P, T);
expx         = exp(x_t)';

N_t      = zeros(T + 1, 1);
N_t(1)   = 1 - uss;
Ks       = zeros(T + 1, 1);
Ks(1)    = K(ceil(nK/2));
% 
V        = zeros(T, 1);
E_t      = zeros(T, 1);
I        = zeros(T, 1);
q_t      = zeros(T, 1);
theta_t  = zeros(T, 1);

for t = 1 : T
    % 
    E_t(t) = funeval( aE(:, index(t)), fspace, {N_t(t), Ks(t)} );
    I(t)   = funeval( aI(:, index(t)), fspace, {N_t(t), Ks(t)} );
    %
    % labor market variables
    q       = kappa0./(E_t(t) - kappa1);
    q       = min(q, 1);
    q(q<0)  = 1;
    q_t(t)  = q;
    %
    theta_t(t) = ( q.^(-iota) - 1 ).^(1/iota);
    V(t)       = theta_t(t).*(1 - N_t(t));
    N_t(t+1)   = (1 - s)*N_t(t) + q_t(t)*V(t);
    N_t(t+1)   = min(Nmax, max(Nmin, N_t(t+1)));
    %
    Ks(t+1)    = (1 - delta)*Ks(t) + ( a1 + (a2/(1 - 1/nu))*((I(t)./Ks(t)).^(1-1/nu)) )*Ks(t); 
    Ks(t+1)    = min(Kmax, max(Kmin, Ks(t+1)));
end

% house cleaning 
N_t(end) = [];
Ks(end)  = [];

ini          = 1 : burnin; 
expx(ini)    = [];
N_t(ini)     = [];
Ks(ini)      = [];
V(ini)       = [];
I(ini)       = [];
E_t(ini)     = [];
q_t(ini)     = [];
theta_t(ini) = [];

% calculating variables of interest
Y        = expx.*(Ks.^alpha).*(N_t.^(1 - alpha));
kappat_t = kappa0 + kappa1*q_t; 
LPY      = (1 - alpha)*Y./N_t;
W        = eta*( LPY + kappat_t.*theta_t ) + (1 - eta)*b;
C        = Y - kappat_t.*V - I;
lambda_t         = zeros(T - burnin, 1); 
lambda_t(q_t==1) = kappa0 + kappa1 - E_t(q_t==1); 
 
% -1 because of return convention with +1 in T
C_t(:, pn)    = C(1 : end-1); 
Y_t(:, pn)    = Y(1 : end-1); 
U_t(:, pn)    = 1 - N_t(1 : end-1);
V_t(:, pn)    = V(1 : end-1); 
LPY_t(:, pn)  = LPY(1 : end-1); 
K_t(:, pn)    = Ks(1 : end-1);
I_t(:, pn)    = I(1 : end-1); 
W_t(:, pn)    = W(1 : end-1);
expx_t(:, pn) = expx(1 : end-1);

end

toc 

nsim = nsim*parnsim; 

U_t    = reshape(U_t, [len nsim]); 
V_t    = reshape(V_t, [len nsim]); 
K_t    = reshape(K_t, [len nsim]); 
I_t    = reshape(I_t, [len nsim]); 
expx_t = reshape(expx_t, [len nsim]); 
LPY_t  = reshape(LPY_t, [len nsim]); 
W_t    = reshape(W_t, [len nsim]); 
C_t    = reshape(C_t, [len nsim]); 
Y_t    = reshape(Y_t, [len nsim]); 

% recalculate theta_t and q_t as intermediate results in parfor loops are lost
theta_t = V_t./U_t;
q_t     = (1 + theta_t.^iota).^(-1/iota); 
f_t     = (1 + theta_t.^(-iota)).^(-1/iota); 

save StatDistr_capital
else
    
load StatDistr_capital
end


 
% construct tab1 across the nsim artificial samples
tab1  = zeros(27, nsim); 
% the length of C and Y sample (annual) in monthly terms
C_t       = C_t(1 : Tc, :);
Y_t       = Y_t(1 : Ty, :);
I_t       = I_t(1 : Ty, :);
%
parfor s = 1 : nsim
    % 
    Ca   = sum(reshape(C_t(:, s), [12 Tc/12]))';
    Ya   = sum(reshape(Y_t(:, s), [12 Ty/12]))';
    Ia   = sum(reshape(I_t(:, s), [12 Ty/12]))';
    % 
    gCa  = log(Ca(2 : end)) - log(Ca(1 : end-1)); 
    gYa  = log(Ya(2 : end)) - log(Ya(1 : end-1));   
    gIa  = log(Ia(2 : end)) - log(Ia(1 : end-1));   
    %
    tab1(:, s) = [
        % C, Y, and I
        std(gCa)*100; 
        skewness(gCa); 
        kurtosis(gCa); 
        myAutoCorr(gCa, [1 2 3 4 5]); 
        std(gYa)*100;
        skewness(gYa); 
        kurtosis(gYa); 
        myAutoCorr(gYa, [1 2 3 4 5]); 
        std(gIa)*100;
        skewness(gIa); 
        kurtosis(gIa); 
        myAutoCorr(gIa, [1 2 3 4 5]); 
        % U
        mean(U_t(1 : Tuv, s))*100;
        skewness(U_t(1 : Tuv, s));
        kurtosis(U_t(1 : Tuv, s));
        ]; 
end

% tab1 output, the I data moments need to be updated 
tab1data = [6.37;  -.55;     9.19;  .0746; .0341; -.0019; -.0201; -.039; 
            5.63;  -1.0153; 11.866; .1568; .0017;  .0215; -.025;  -.0159; 
            5.63;  -1.0153; 11.866; .1568; .0017;  .0215; -.025;  -.0159; 
            7.12;   1.99;   6.82
            ];

tab1pval = zeros(27, 1);
for row  = 1 : 27
    tab1pval(row) = sum( tab1(row, :) > tab1data(row) )/nsim;
end
tab1out = [mean(tab1, 2) prctile(tab1', [5 95])' tab1pval] 


% constructing tab2 across the nsim artificial samples
tab2 = zeros(6, 4, nsim);
%
e_wp1 = zeros(nsim, 1); % the elasticity of wage with respect to productivity
e_wp2 = zeros(nsim, 1); % the elasticity of wage with respect to productivity
%
hp_smooth = 1600;
parfor s = 1 : nsim 
    % take quarterly averages of monthly observations
    LPY_q     = mean(reshape(LPY_t(1 : Tuv, s), [3 Tuv/3]))';
    U_q       = mean(reshape(U_t(1 : Tuv, s), [3 Tuv/3]))'; 
    V_q       = mean(reshape(V_t(1 : Tuv, s), [3 Tuv/3]))';
    W_q       = mean(reshape(W_t(1 : Tuv, s), [3 Tuv/3]))';
    theta_q   = V_q./U_q; 
    %
    U_hp     = (U_q - mean(U_q))/mean(U_q); 
    Ud       = U_hp - hpfilter(U_hp, hp_smooth); 
    % 
    V_hp     = (V_q - mean(V_q))/mean(V_q); 
    Vd       = V_hp - hpfilter(V_hp, hp_smooth); 
    % 
    theta_hp = (theta_q - mean(theta_q))/mean(theta_q); 
    thetad   = theta_hp - hpfilter(theta_hp, hp_smooth); 
    % 
    LPY_hp   = (LPY_q - mean(LPY_q))/mean(LPY_q); 
    LPYd     = LPY_hp - hpfilter(LPY_hp, hp_smooth);
    %
    tab2(:, :, s) = [std(Ud) std(Vd) std(thetad) std(LPYd);
        myAutoCorr(Ud, [1]) myAutoCorr(Vd, [1]) myAutoCorr(thetad, [1]) myAutoCorr(LPYd, [1]);
        corrcoef([Ud Vd thetad LPYd])]; 
    % 
    % calculate the elasticity of wage with respect to productivity
    pdlog = log(LPY_q) - hpfilter(log(LPY_q), hp_smooth); 
    wdlog = log(W_q) - hpfilter(log(W_q), hp_smooth); 
    [coefs, tstats, R2, chi2, pval] = MultiRegressNw(wdlog, pdlog, [], 0, 1);
    e_wp1(s) = coefs(2);
    % 
    p_hp  = (LPY_q - mean(LPY_q))/mean(LPY_q);
    p_d   = p_hp - hpfilter(p_hp, hp_smooth);
    w_hp  = (W_q - mean(W_q))/mean(W_q);
    w_d   = w_hp - hpfilter(w_hp, hp_smooth);
    [coefs, tstats, R2, chi2, pval] = MultiRegressNw(w_d, p_d, [], 0, 1);
    e_wp2(s) = coefs(2);
end
%
% tab2 output
tab2mean  = mean(tab2, 3)
tab2std   = std(tab2, 0, 3)   
% 
e_wp1mean = mean(e_wp1)
e_wp1std  = std(e_wp1)
e_wp2mean = mean(e_wp2)
e_wp2std  = std(e_wp2)


% close parallel environment
if needNewWorkers
    matlabpool close
end

