% PARSIMU_DISASTER_CAPITAL:
% Charaterize the disaster moments in the search economy with capital
% 
%  Nicolas Petrosky-Nadeau, Lu Zhang, and Lars Kuehn, January 2018


clear all; close all; clc; format compact; format short; 


% Open parallel environment
needNewWorkers = (matlabpool('size') == 0);
if needNewWorkers    
    % Open a new MATLAB pool with 16 workers.
    matlabpool open 16
end


load output_proj_capital
load moments_theory_capital

[kappa0, kappa1, eta, b, s, iota, xbar, rhox, stdx, beta, alpha, delta, a1, a2, nu] = params{:};
[P, X, N, K, NK, fspace, Nm, Km, Xm, Xp] = gridsetup{:};

[uss, nss, qss, fss, thetass, yss, vss, css, wss, Jss, kss, kappatss] = SS{:}; 

nN      = length(N);
nK      = length(K);
nNK     = nN*nK;
nX      = length(X);
%
Nmin    = min(N);
Nmax    = max(N);
Kmin    = min(K);
Kmax    = max(K);


burnin  = 500*12; 
parnsim = 100
nsim    = 100 
len     = 145*12;

% match Barro and Ursua's sample size averaged across C and Y
T       = burnin + nsim*len; 


if 1 % change 1 to 0 if you have run the code once and saved the stationary distribution 

C_t    = zeros(len*nsim, parnsim); 
Y_t    = zeros(len*nsim, parnsim); 

tic; 
parfor pn = 1 : parnsim

% simulating artificial data in one shot
x            = log(X); 
[x_t, index] = DiscSimu(median(x), x, P, T);
expx_t       = exp(x_t)';

N_t      = zeros(T + 1, 1);
N_t(1)   = 1 - uss;

K_t      = zeros(T + 1, 1);
K_t(1)   = K(ceil(nK/2));

V_t      = zeros(T, 1);
I_t      = zeros(T, 1);
q_t      = zeros(T, 1);

for t = 1 : T
    E_t    = funeval( aE(:, index(t)), fspace, { N_t(t), K_t(t) } );
    I_t(t) = funeval( aI(:, index(t)), fspace, { N_t(t), K_t(t) } );
    %
    % labor market variables
    q       = kappa0./(E_t - kappa1);
    q       = min(q, 1);
    q(q<0)  = 1;
    q_t(t)  = q;
    %
    theta_t     = ( q.^(-iota) - 1 ).^(1/iota);
    V_t(t)      = theta_t.*(1 - N_t(t));
    N_t(t+1)    = (1 - s)*N_t(t) + q_t(t)*V_t(t);
    N_t(t+1)    = min( Nmax, max(Nmin, N_t(t+1)) );
    % 
    K_t(t+1)    = (1 - delta)*K_t(t) + ( a1 + (a2/(1 - 1/nu))*((I_t(t)./K_t(t)).^(1-1/nu)) )*K_t(t); 
    K_t(t+1)    = min( Kmax, max(Kmin, K_t(t+1)) );
    %
end

% house cleaning 
N_t(end) = [];
K_t(end) = [];

ini         = 1 : burnin; 
expx_t(ini) = [];
N_t(ini)    = [];
K_t(ini)    = [];
V_t(ini)    = [];
I_t(ini)    = [];
q_t(ini)    = [];

% calculating variables of interest
Y_t(:, pn) = expx_t.*(K_t.^alpha).*(N_t.^(1 - alpha));
C_t(:, pn) = Y_t(:, pn) - (kappa0 + kappa1*q_t).*V_t - I_t;

end

nsim = nsim*parnsim; 

C_t    = reshape(C_t, [len nsim]); 
Y_t    = reshape(Y_t, [len nsim]); 

save disaster_capital
else
load disaster_capital
end


% split the long simulated data by 1000 economies
Tcy  = len;

% initializing
disaster_prob_C = zeros(nsim, 1);        % disaster probability: C 
disaster_size_C = zeros(nsim, 1);        % disaster mean size: C 
disaster_dura_C = zeros(nsim, 1);        % disaster mean duration: C 
% 
disaster_prob_Y = zeros(nsim, 1);        % disaster probability: Y
disaster_size_Y = zeros(nsim, 1);        % disaster mean size: Y
disaster_dura_Y = zeros(nsim, 1);        % disaster mean duration: Y
% 
for s = 1 : nsim
    
    % Time-aggregate
    C   = sum(reshape(C_t(:, s), [12 Tcy/12]))';
    Y   = sum(reshape(Y_t(:, s), [12 Tcy/12]))';
     
    % Compute annual consumption and GDP growth for the s-th economy
    gC  = log(C(2 : end)) - log(C(1 : end-1)); 
    gY  = log(Y(2 : end)) - log(Y(1 : end-1));   
    
    % Apply the peak-to-trough measurement for consumption disasters
    gCneg    = find(gC < 0);      % find index of negative C growth 
    disC     = [];                % dynamic matrix: each row is each disaster, and the two columns are size and duration of the disaster
    p_peak   = 1;
    p_trough = p_peak;
    while p_trough < length(gCneg), 
        while gCneg(p_trough + 1) == gCneg(p_trough) + 1,   
            p_trough = p_trough + 1; 
            if p_trough >= length(gCneg)
                break
            end
        end
        dsize = sum(gC(gCneg(p_peak : p_trough))); 
        if abs(dsize) > 0.10,
            disC = [disC; [dsize p_trough - p_peak + 1]]; 
        end
        p_peak   = p_trough + 1; 
        p_trough = p_peak; 
    end
    
    % Apply the peak-to-trough measurement for GDP disasters
    gYneg    = find(gY < 0);      % find index of negative Y growth 
    disY     = [];                % dynamic matrix: each row is each disaster, and the two columns are size and duration of the disaster
    p_peak   = 1;
    p_trough = p_peak;
    while p_trough < length(gYneg), 
        while gYneg(p_trough + 1) == gYneg(p_trough) + 1,   
            p_trough = p_trough + 1; 
            if p_trough >= length(gYneg)
                break
            end
        end
        dsize = sum(gY(gYneg(p_peak : p_trough))); 
        if abs(dsize) > 0.10,
            disY = [disY; [dsize p_trough - p_peak + 1]]; 
        end
        p_peak   = p_trough + 1; 
        p_trough = p_peak; 
    end
     
    % Calculate consumption disaster statistics for the s-th economy
    if length(disC) > 0, 
        % general statistics
        disaster_prob_C(s) = size(disC, 1)/(Tcy/12 - sum(disC(:, 2))); 
        disaster_size_C(s) = mean(disC(:, 1)); 
        disaster_dura_C(s) = mean(disC(:, 2)); 
    else
        disp(['no disasters in the simulated economy #' num2str(s) '!'])
    end
    
    % Calculate output disaster statistics for the s-th economy
    if length(disY) > 0, 
        % general statistics
        disaster_prob_Y(s) = size(disY, 1)/(Tcy/12 - sum(disY(:, 2))); 
        disaster_size_Y(s) = mean(disY(:, 1)); 
        disaster_dura_Y(s) = mean(disY(:, 2)); 
    else
        disp(['no output disasters in the simulated economy #' num2str(s) '!'])
    end
    
end

% Report the overall results
pC_std     = std(disaster_prob_C)
pY_std     = std(disaster_prob_Y)

ndindex_C  = find(disaster_size_C == 0);
disaster_size_C(ndindex_C, :) = [];
disaster_dura_C(ndindex_C, :) = [];

ndindex_Y  = find(disaster_size_Y == 0);
disaster_size_Y(ndindex_Y, :) = [];
disaster_dura_Y(ndindex_Y, :) = [];

taboutC   = [mean(disaster_prob_C)  prctile(disaster_prob_C, [5 95])  sum(disaster_prob_C > .0857)/nsim;
             mean(-disaster_size_C) prctile(-disaster_size_C, [5 95]) sum(abs(disaster_size_C) > .2316)/nsim; 
             mean(disaster_dura_C)  prctile(disaster_dura_C, [5 95])  sum(disaster_dura_C > 3.75)/nsim
            ]

taboutY   = [mean(disaster_prob_Y)  prctile(disaster_prob_Y, [5 95])  sum(disaster_prob_Y > .0783)/nsim;
             mean(-disaster_size_Y) prctile(-disaster_size_Y, [5 95]) sum(abs(disaster_size_Y) > .2199)/nsim; 
             mean(disaster_dura_Y)  prctile(disaster_dura_Y, [5 95])  sum(disaster_dura_Y > 3.72)/nsim
            ]



% close parallel environment
if needNewWorkers
    matlabpool close
end






