
function err = residual_capital(a0, params, gridsetup)

% Compute the error functions in the capital model 
% 
%  Nicolas Petrosky-Nadeau, Lu Zhang, and Lars Kuehn, January 2018


[kappa0, kappa1, eta, b, s, iota, xbar, rhox, stdx, beta, alpha, delta, a1, a2, nu] = params{:};
[P, X, N, K, NK, fspace, Nm, Km, Xm, Xp] = gridsetup{:};

nN      = length(N); 
nK      = length(K);
nNK     = nN*nK;
nX      = length(X);
%
Nmin    = min(N); Nmax    = max(N);
Kmin    = min(K); Kmax    = max(K);

% Policy functions 
aE      = reshape(a0(1 : nN*nK*nX), [nN*nK nX]);
aI      = reshape(a0(  nN*nK*nX + 1 : 2*nN*nK*nX), [nN*nK nX]);
%
E       = funeval(aE, fspace, NK);
I       = funeval(aI, fspace, NK);

% Current period variables
q       = kappa0./(E - kappa1); 
q       = min(q, 1);
q(q<0)  = 1;
kappat  = kappa0 + kappa1*q; 
theta   = ( q.^(-iota) - 1 ).^(1/iota);
% 
V       = theta.*(1 - Nm);
Y       = Xm.*(Km.^alpha).*(Nm.^(1 - alpha)); 
C       = Y - kappat.*V - I;

% Next period variables that are independent of X(t+1) - nN*nK by nX
Np      = (1 - s)*Nm + q.*V;
Np      = min( Nmax, max(Nmin, Np) );
Kp      = (1 - delta).*Km + ( a1 + (a2/(1 - 1/nu))*((I./Km).^(1-1/nu)) ).*Km; 
Kp      = min( Kmax, max(Kmin, Kp) );

% Next period variables that depend on X(t+1)
Ep      = zeros(nNK, nX, nX);
Ip      = zeros(nNK, nX, nX);
NK_Resh = reshape([Np Kp], [nNK*nX 2]);

parfor ip = 1 : nX         % given x_t+1 = x(ip)
    Ep(:, :, ip) = reshape(funeval( aE(:, ip), fspace, NK_Resh ), [nNK nX]);
    Ip(:, :, ip) = reshape(funeval( aI(:, ip), fspace, NK_Resh ), [nNK nX]);
end


% Build job creation and investment Euler equations errors
qp       = kappa0./(Ep - kappa1);
qp       = min(1, qp);
qp(qp<0) = 1;
kappatp  = kappa0 + kappa1*qp; 
thetap   = ( qp.^(-iota) - 1 ).^(1/iota);
% 
lambdap        = zeros(nNK, nX, nX);
lambdap(qp==1) = kappa0 + kappa1 - Ep(qp==1);

% Recall the grid is now (nN*nK by nX by nX)
Npm     = repmat(Np, [1 1 nX]);
Kpm     = repmat(Kp, [1 1 nX]);
Cm      = repmat(C,  [1 1 nX]);
% 
Up        = 1 - Npm;
Vp        = thetap.*Up;
Vp(qp==1) = 0;
% 
Yp      = Xp.*(Kpm.^alpha).*(Npm.^(1 - alpha));
Cp      = Yp - kappatp.*Vp - Ip; 
% 
Wp      = eta*( (1 - alpha)*Yp./Npm + kappatp.*thetap ) + (1 - eta)*b; 
Mp      = beta * (Cm./Cp);

% Euler equation for vacancies
inside_JC = Mp.*( (1 - alpha)*Yp./Npm - Wp + (1 - s)*(kappatp./qp - lambdap) ); 
% 
rhs_JC  = zeros(nNK, nX);
for ix  = 1 : nX
    rhs_JC(:, ix) = squeeze(inside_JC(:, ix, :)) * P(:, ix);
end
% 
eV      = E - rhs_JC;

% Euler equation for investment
inside_I = Mp.*( alpha*Yp./Kpm + (1/a2)*((Ip./Kpm).^(1/nu))*(1 - delta + a1) + (1/(nu-1))*(Ip./Kpm) ); 
% 
rhs_I  = zeros(nNK, nX);
for ix = 1 : nX
    rhs_I(:, ix) = squeeze(inside_I(:, ix, :)) * P(:, ix);
end
% 
lhs_I  = (1/a2)*( (I./Km).^(1/nu) );
eI     = lhs_I - rhs_I;

err    = [eV(:); eI(:)];

