% PARSIMU_DISASTER_RECURSIVE_UTILITY: 
% Calculate the recursive utility model's disaster moments via 10,000 simulations
% 
%  Nicolas Petrosky-Nadeau, Lu Zhang, and Lars Kuehn, January 2018


clear all; close all; clc; format compact; format short; 

% Open parallel environment
needNewWorkers = (matlabpool('size') == 0);
if needNewWorkers    
    % Open a new MATLAB pool with 16 workers.
    matlabpool open 16
end


load output_proj_recursive_utility
load moments_theory_recursive_utility

[kappa0, kappa1, eta, b, s, nu, rhox, stdx, beta, gamma, psi] = params{:};
alpha   = 1 - gamma;
rho     = 1 - 1/psi;

[P, x, Xm, X_p, N, Nm, fspace] = gridsetup{:};
[uss, qss, fss, thetass, yss, vss, css, wss] = SS{:};

nn      = length(N); 
nx      = length(x);
Nmin    = min(N);
Nmax    = max(N);

burnin  = 500*12; 
parnsim = 100
nsim    = 100 
len     = 145*12;

% match Barro and Ursua's sample size averaged across C and Y
T       = burnin + nsim*len; 


if 1 % change 1 to 0 if you have run the code once and saved the stationary distribution 
C_t    = zeros(len*nsim, parnsim); 
Y_t    = zeros(len*nsim, parnsim); 

tic; 
parfor pn = 1 : parnsim
    
% simulating artificial data in one shot
[x_t, index] = DiscSimu(median(x), x, P, T);
expx_t       = exp(x_t)';

N_t      = zeros(T + 1, 1);
N_t(1)   = 1 - uss;

V_t      = zeros(T, 1);
q_t      = zeros(T, 1);

tic;
for t = 1 : T
    E_t = funeval(aE(:, index(t)), fspace, N_t(t));
    %
    % labor market variables
    q       = kappa0./(E_t - kappa1);
    q       = min(q,1);
    q(q<0)  = 1;
    q_t(t)  = q;
    %
    theta_t     = ( q.^(-nu) - 1 ).^(1/nu);
    V_t(t)      = theta_t.*(1 - N_t(t));
    N_t(t+1)    = (1 - s)*N_t(t) + q_t(t)*V_t(t);
    N_t(t+1)    = min(Nmax, max(Nmin, N_t(t+1)));
    % 
end

% house cleaning 
N_t(end)     = [];
ini          = 1 : burnin; 
expx_t(ini)  = [];
N_t(ini)     = [];
V_t(ini)     = [];
q_t(ini)     = [];

% calculating variables of interest
Y_t(:, pn)   = expx_t.*N_t;
C_t(:, pn)   = expx_t.*N_t - (kappa0 + kappa1*q_t).*V_t;

end

nsim = nsim*parnsim; 

C_t    = reshape(C_t, [len nsim]); 
Y_t    = reshape(Y_t, [len nsim]); 


save disaster_recursive_utility
else
load disaster_recursive_utility
end


% split the long simulated data by 1000 economies
Tcy  = len;

% initializing
disaster_prob_C = zeros(nsim, 1);        % disaster probability: C 
disaster_size_C = zeros(nsim, 1);        % disaster mean size: C 
disaster_dura_C = zeros(nsim, 1);        % disaster mean duration: C 
% 
disaster_prob_Y = zeros(nsim, 1);        % disaster probability: Y
disaster_size_Y = zeros(nsim, 1);        % disaster mean size: Y
disaster_dura_Y = zeros(nsim, 1);        % disaster mean duration: Y
% 
for s = 1 : nsim
    
    % Time-aggregate
    C   = sum(reshape(C_t(:, s), [12 Tcy/12]))';
    Y   = sum(reshape(Y_t(:, s), [12 Tcy/12]))';
     
    % Compute annual consumption and GDP growth for the s-th economy
    gC  = log(C(2 : end)) - log(C(1 : end - 1)); 
    gY  = log(Y(2 : end)) - log(Y(1 : end - 1));   
    
    % Apply the peak-to-trough measurement for consumption disasters
    gCneg    = find(gC < 0);      % find index of negative C growth 
    disC     = [];                % dynamic matrix: each row is each disaster, and the two columns are size and duration of the disaster
    p_peak   = 1;
    p_trough = p_peak;
    while p_trough < length(gCneg), 
        while gCneg(p_trough + 1) == gCneg(p_trough) + 1,   
            p_trough = p_trough + 1; 
            if p_trough >= length(gCneg)
                break
            end
        end
        dsize = sum(gC(gCneg(p_peak : p_trough))); 
        if abs(dsize) > 0.10,
            disC = [disC; [dsize p_trough - p_peak + 1]]; 
        end
        p_peak   = p_trough + 1; 
        p_trough = p_peak; 
    end
    
    % Apply the peak-to-trough measurement for GDP disasters
    gYneg    = find(gY < 0);      % find index of negative Y growth 
    disY     = [];                % dynamic matrix: each row is each disaster, and the two columns are size and duration of the disaster
    p_peak   = 1;
    p_trough = p_peak;
    while p_trough < length(gYneg), 
        while gYneg(p_trough + 1) == gYneg(p_trough) + 1,   
            p_trough = p_trough + 1; 
            if p_trough >= length(gYneg)
                break
            end
        end
        dsize = sum(gY(gYneg(p_peak : p_trough))); 
        if abs(dsize) > 0.10,
            disY = [disY; [dsize p_trough - p_peak + 1]]; 
        end
        p_peak   = p_trough + 1; 
        p_trough = p_peak; 
    end
     
    % Calculate consumption disaster statistics for the s-th economy
    if length(disC) > 0, 
        % general statistics
        disaster_prob_C(s) = size(disC, 1)/(Tcy/12 - sum(disC(:, 2))); 
        disaster_size_C(s) = mean(disC(:, 1)); 
        disaster_dura_C(s) = mean(disC(:, 2)); 
    else
        disp(['no consumption disasters in the simulated economy #' num2str(s) '!'])
    end

    % Calculate output disaster statistics for the s-th economy
    if length(disY) > 0, 
        % general statistics
        disaster_prob_Y(s) = size(disY, 1)/(Tcy/12 - sum(disY(:, 2))); 
        disaster_size_Y(s) = mean(disY(:, 1)); 
        disaster_dura_Y(s) = mean(disY(:, 2)); 
    else
        disp(['no output disasters in the simulated economy #' num2str(s) '!'])
    end
        
end

% Report the overall results
pC_std     = std(disaster_prob_C)
pY_std     = std(disaster_prob_Y)

ndindex_C  = find(disaster_size_C == 0);
disaster_size_C(ndindex_C, :) = [];
disaster_dura_C(ndindex_C, :) = [];

ndindex_Y  = find(disaster_size_Y == 0);
disaster_size_Y(ndindex_Y, :) = [];
disaster_dura_Y(ndindex_Y, :) = [];

taboutC   = [mean(disaster_prob_C)  prctile(disaster_prob_C, [5 95])  sum(disaster_prob_C > .0857)/nsim;
             mean(-disaster_size_C) prctile(-disaster_size_C, [5 95]) sum(abs(disaster_size_C) > .2316)/nsim; 
             mean(disaster_dura_C)  prctile(disaster_dura_C, [5 95])  sum(disaster_dura_C > 3.75)/nsim
            ]

taboutY   = [mean(disaster_prob_Y)  prctile(disaster_prob_Y, [5 95])  sum(disaster_prob_Y > .0783)/nsim;
             mean(-disaster_size_Y) prctile(-disaster_size_Y, [5 95]) sum(abs(disaster_size_Y) > .2199)/nsim; 
             mean(disaster_dura_Y)  prctile(disaster_dura_Y, [5 95])  sum(disaster_dura_Y > 3.72)/nsim
             ]

         
% close parallel environment
if needNewWorkers
    matlabpool close
end





