% PARSIMU:
% Calculate all the model moments via 10,000 simulations 
% 
%  Nicolas Petrosky-Nadeau, Lu Zhang, and Lars Kuehn, January 2018


clear all; close all; clc; format compact; format short; 


% Open parallel environment
needNewWorkers = (matlabpool('size') == 0);
if needNewWorkers    
    % Open a new MATLAB pool with 16 workers, reduce the number if you don't have that many workers on your computer
    matlabpool open 16
end


load output_proj_log_utility
load moments_theory_log_utility

[kappa0, kappa1, eta, b, s, nu, rhox, stdx, beta] = params{:};

[P, x, Xm, X_p, N, Nm, fspace] = gridsetup{:};
[uss, qss, fss, thetass, yss, vss, css, wss] = SS{:};

nn      = length(N); 
nx      = length(x);
Nmin    = min(N);
Nmax    = max(N);

burnin  = 500*12; 
parnsim = 100
nsim    = 100 

% average length of AP data in months
Tap     = 124*12;                                   
Tc      = 138*12;
Ty      = 153*12; 
Tuv     = 1017; 
len     = max([Tap Tc Ty Tuv]); 
% 
% + 1 to make sure (excess) returns have nsim*len periods
T       = burnin + nsim*len + 1


if 1 % set 1 to 0 if you have run the code once and saved the (large) distribution file
U_t    = zeros(len*nsim, parnsim); 
V_t    = zeros(len*nsim, parnsim); 
expx_t = zeros(len*nsim, parnsim);  
W_t    = zeros(len*nsim, parnsim);  
C_t    = zeros(len*nsim, parnsim); 
Y_t    = zeros(len*nsim, parnsim); 

tic; 
parfor pn = 1 : parnsim

% simulating artificial data in one shot
[x_t, index] = DiscSimu(median(x), x, P, T);
expx         = exp(x_t)';

N_t      = zeros(T + 1, 1);
N_t(1)   = 1 - uss;

V        = zeros(T, 1);
E_t      = zeros(T, 1);
q_t      = zeros(T, 1);
theta_t  = zeros(T, 1);

for t = 1 : T
    E_t(t) = funeval(aE(:, index(t)), fspace, N_t(t));
    %
    % labor market variables
    q       = kappa0./(E_t(t) - kappa1);
    q       = min(q,1);
    q(q<0)  = 1;
    q_t(t)  = q;
    %
    theta_t(t)  = ( q.^(-nu) - 1 ).^(1/nu);
    V(t)        = theta_t(t).*(1 - N_t(t));
    N_t(t+1)    = (1 - s)*N_t(t) + q_t(t)*V(t);
    N_t(t+1)    = min(Nmax, max(Nmin, N_t(t+1)));
end

% house cleaning 
N_t(end)      = [];
ini           = 1 : burnin; 
expx(ini)     = [];
N_t(ini)      = [];
V(ini)        = [];
E_t(ini)      = [];
q_t(ini)      = [];
theta_t(ini)  = [];

% calculating variables of interest
Y             = expx.*N_t;
W             = eta*(expx + (kappa0 + kappa1*q_t).*theta_t) + (1 - eta)*b;
C             = Y - (kappa0 + kappa1*q_t).*V;
lambda_t      = zeros(T - burnin, 1); 
lambda_t(q_t==1) = kappa0 + kappa1 - E_t(q_t==1); 
% 
C_t(:, pn)    = C(1 : end-1); 
Y_t(:, pn)    = Y(1 : end-1); 
% 
U_t(:, pn)    = 1 - N_t(1 : end-1);
W_t(:, pn)    = W(1 : end-1);
V_t(:, pn)    = V(1 : end-1); 
expx_t(:, pn) = expx(1 : end-1);

end

toc 

nsim = nsim*parnsim; 


U_t    = reshape(U_t, [len nsim]); 
V_t    = reshape(V_t, [len nsim]); 
expx_t = reshape(expx_t, [len nsim]); 
W_t    = reshape(W_t, [len nsim]); 
%
C_t    = reshape(C_t, [len nsim]); 
Y_t    = reshape(Y_t, [len nsim]); 

save StatDistr_log_utility
else
load StatDistr_log_utility
end



% construct C and Y moments across the nsim artificial samples
tab1 = zeros(19, nsim); 
% the length of C and Y sample (annual) in monthly terms
C_t       = C_t(1 : Tc, :);
Y_t       = Y_t(1 : Ty, :);
%
parfor s = 1 : nsim
    % 
    Ca   = sum(reshape(C_t(:, s), [12 Tc/12]))';
    Ya   = sum(reshape(Y_t(:, s), [12 Ty/12]))';
    gCa  = log(Ca(2 : end)) - log(Ca(1 : end-1)); 
    gYa  = log(Ya(2 : end)) - log(Ya(1 : end-1));   
    %
    tab1(:, s) = [
        % C and Y
        std(gCa)*100; 
        skewness(gCa); 
        kurtosis(gCa); 
        myAutoCorr(gCa, [1 2 3 4 5]); 
        std(gYa)*100;
        skewness(gYa); 
        kurtosis(gYa); 
        myAutoCorr(gYa, [1 2 3 4 5]); 
        % U
        mean(U_t(1 : Tuv, s))*100;
        skewness(U_t(1 : Tuv, s));
        kurtosis(U_t(1 : Tuv, s));
        ]; 
    % 
end

% tab1 output
tab1data = [6.37; -.55;      9.19; .0746; .0341; -.0019; -.0201; -.039; 
            5.63; -1.0153; 11.866; .1568; .0017; .0215;  -.025; -.0159; 
            7.12; 1.99; 6.82];
        
tab1pval = zeros(19, 1);
for row  = 1 : 19
    tab1pval(row) = sum( tab1(row, :) > tab1data(row) )/nsim;
end
tab1out = [mean(tab1, 2) prctile(tab1',[5 95])' tab1pval] 



% construct labor market moments across the nsim artificial samples
tab2  = zeros(6, 4, nsim);
% 
% the elasticity of wage with respect to productivity
e_wp1 = zeros(nsim, 1); 
e_wp2 = zeros(nsim, 1); 
%
hp_smooth = 1600;
detrend   = 1; 
parfor s  = 1 : nsim 
    % 
    % take quarterly averages of monthly observations
    expx_q    = mean(reshape(expx_t(1 : Tuv, s), [3 Tuv/3]))';
    U_q       = mean(reshape(U_t(1 : Tuv, s), [3 Tuv/3]))'; 
    V_q       = mean(reshape(V_t(1 : Tuv, s), [3 Tuv/3]))';
    W_q       = mean(reshape(W_t(1 : Tuv, s), [3 Tuv/3]))';
    theta_q   = V_q./U_q; 
    %
    U_hp      = (U_q - mean(U_q))/mean(U_q); 
    Ud        = U_hp - hpfilter(U_hp, hp_smooth); 
    V_hp      = (V_q - mean(V_q))/mean(V_q); 
    Vd        = V_hp - hpfilter(V_hp, hp_smooth); 
    theta_hp  = (theta_q - mean(theta_q))/mean(theta_q); 
    thetad    = theta_hp - hpfilter(theta_hp, hp_smooth); 
    X_hp      = (expx_q - mean(expx_q))/mean(expx_q); 
    Xd        = X_hp - hpfilter(X_hp, hp_smooth); 
    %
    tab2(:, :, s) = [std(Ud) std(Vd) std(thetad) std(Xd);
        myAutoCorr(Ud, [1]) myAutoCorr(Vd, [1]) myAutoCorr(thetad, [1]) myAutoCorr(Xd, [1]);
        corrcoef([Ud Vd thetad Xd])]; 
    % 
    % calculate the elasticity of wage with respect to productivity
    pdlog = log(expx_q) - hpfilter(log(expx_q), hp_smooth); 
    wdlog = log(W_q) - hpfilter(log(W_q), hp_smooth); 
    % 
    [coefs, tstats, R2, chi2, pval] = MultiRegressNw(wdlog, pdlog, [], 0, 1);
    e_wp1(s) = coefs(2);
    % 
    p_hp  = (expx_q - mean(expx_q))/mean(expx_q);
    p_d   = p_hp - hpfilter(p_hp, hp_smooth);
    w_hp  = (W_q - mean(W_q))/mean(W_q);
    w_d   = w_hp - hpfilter(w_hp, hp_smooth);
    % 
    [coefs, tstats, R2, chi2, pval] = MultiRegressNw(w_d, p_d, [], 0, 1);
    e_wp2(s) = coefs(2);
    % 
end
%
% tab2 output
tab2mean  = mean(tab2, 3)
tab2std   = std(tab2, 0, 3)   

Uvol      = squeeze(tab2(1, 1, :)); 
mUvol     = mean(Uvol)
prcUvol   = prctile(Uvol, [5 95])
pvalUv    = sum(Uvol > .1983)/nsim

% 
e_wp1mean = mean(e_wp1)
e_wp1std  = std(e_wp1)
e_wp2mean = mean(e_wp2)
e_wp2std  = std(e_wp2)


% close parallel environment
if needNewWorkers
    matlabpool close
end

