% PROJECTION: 
% Use projection methods to solve the endogenous disasters model with V >= 0 imposed via Christiano and Fisher (2000) method
% 
%  Nicolas Petrosky-Nadeau, Lu Zhang, and Lars Kuehn, January 2018


clear all; close all; clc; format compact; format long

% time discount factor adjusted to match the mean discount rate in international data, 5.73% per annum  
beta    = exp( -(log(1.0573))/12 )               
eta     = .04                                       % worker's bargaining weight 
b       = .85                                       % value of unemployment benefit    
rhox    = .95^(1/3)
stdx    = .01
% 
qss     = .71;                 
fss     = .45;
uss     = .10;
s       = .04
ustdx   = stdx/sqrt(1 - rhox^2);
nss     = 1 - uss;
thetass = fss/qss;
err     = @(x)qss - (thetass^x + 1)^(-1/x); 
nu      = 1.25

kappa0  = .5
kappa1  = .5

wss     = eta*(1 + (kappa0 + kappa1*qss)*thetass) + (1 - eta)*b;
vss     = thetass*uss;
yss     = 1 - uss;
css     = yss - (kappa0 + kappa1*qss)*vss;
SS      = {uss, qss, fss, thetass, yss, vss, css, wss};
% 
params  = {kappa0, kappa1, eta, b, s, nu, rhox, stdx, beta};


% discretize aggregate shocks via the Rouwenhorst (1995) method
nx        = 17                                        % the number of grid points for x, a column vector
xdev      = 2*stdx/sqrt((1 - rhox^2)*(nx - 1));
[P, x]    = rouwTrans(rhox, 0, xdev, nx);             % Qx_ij is the probability of jumping to state i next period given the current state of j

% ground work for cubic splines
nn        = 75                                        % # of elements in splines = number of N grid points
Nmin      = .05         
Nmax      = .99
fspace    = fundefn('spli', nn, Nmin, Nmax);
N         = funnode(fspace);

% next period X grid 
Xm        = repmat(exp(x)', [nn 1]);
X_p       = permute(repmat(Xm, [1 1 nx]), [1 3 2]);
% a more intuitive construction of X_p:
% X_p     = zeros(nn, nx, nx);
% for ip  = 1 : nx
%    X_p(:, :, ip) = exp(x(ip))*ones(nn, nx);  
% end
Nm        = repmat(N, [1 nx]);
gridsetup = {P, x, Xm, X_p, N, Nm, fspace};


% load the initial condition (optimally solved) 
load output_proj_log_utility  aE

a0 = [aE(:)];

precision = 0
if precision == 0
    opts = optimset('Display', 'Iter', 'MaxIter', 250);
elseif precision == 1
    opts = optimset('Display', 'Iter', 'MaxIter', 100, 'TolFun', 1e-12, 'TolX', 1e-12);
end
[a, fval, exitflag]  = fsolve('residual_log_utility', a0, opts, params, gridsetup);

exitflag

aE      = reshape(a(1 : nn*nx), [nn nx]);

if exitflag == 1
    save output_proj_log_utility  aE params gridsetup SS
end


if 0  % change 0 to 1 if you want to see some plots 
E      = real(funeval(aE, fspace, N));

q      = kappa0./(E - kappa1); 
q      = min(q, 1); 
q(q<0) = 1; 
theta  = ( q.^(-nu) - 1 ).^(1/nu);
f      = theta.*q; 
V      = theta.*(1 - Nm); 
C      = Xm.*Nm - (kappa0 + kappa1*q).*V; 

figure; surf(exp(x),N,E); title('E')
figure; surf(exp(x),N,q); title('q')
figure; surf(exp(x),N,theta); title('theta')
figure; surf(exp(x),N,V); title('V')
figure; surf(exp(x),N,C); title('C')
end

